//Class which buffers file access.   Lacks the extra features provided by BufferedFile, but this is useful for
//buffering output to a text file or other file formats where the extra byte that BufferedFile sticks on the
//front of the file to indicate endianness interferes with the output file.

//PlainBufferedFile still provides all of the functionality of a BFile: It now provides Read/Write access, random
//access, etc.  All the regular BFile calls can be mixed with the PlainBufferedFile calls, although the BFile calls
//won't benefit from the buffering (they force the buffer to flush, and actually introduce inefficiency - use
//them carefully; in general try to stick to the BufferedRead and BufferedWrite calls)


//******************************************************************************************************
//**** SYSTEM HEADER FILES
//******************************************************************************************************
#include <stdio.h>


//******************************************************************************************************
//**** PROJECT HEADER FILES
//******************************************************************************************************
#include "PlainBufferedFile.h"


//******************************************************************************************************
//**** PlainBufferedFile CLASS DEFINITION
//******************************************************************************************************
PlainBufferedFile::PlainBufferedFile(const entry_ref* ref, uint32 open_Mode, int32 Buffer_Size)
: BFile(ref,open_Mode)
{
	openMode = open_Mode;
	CheckNewFileStatus();
	AllocAndInitBuffer(Buffer_Size);
}


PlainBufferedFile::PlainBufferedFile(const BEntry* entry, uint32 open_Mode, int32 Buffer_Size)
: BFile(entry,open_Mode)
{
	openMode = open_Mode;
	CheckNewFileStatus();
	AllocAndInitBuffer(Buffer_Size);
}


PlainBufferedFile::PlainBufferedFile(const char* path, uint32 open_Mode, int32 Buffer_Size)
: BFile(path,open_Mode)
{
	openMode = open_Mode;
	CheckNewFileStatus();
	AllocAndInitBuffer(Buffer_Size);
}


PlainBufferedFile::PlainBufferedFile(BDirectory* dir, const char* path, uint32 open_Mode, int32 Buffer_Size)
: BFile(dir,path,open_Mode)
{
	openMode = open_Mode;
	CheckNewFileStatus();
	AllocAndInitBuffer(Buffer_Size);
}


void PlainBufferedFile::CheckNewFileStatus()
{
	//Check the status of the file
	Valid = BFile::InitCheck();
}


void PlainBufferedFile::AllocAndInitBuffer(int32 Buffer_Size)
{
	Buffer = NULL;
	if(Valid == B_NO_ERROR)
	{
		//Allocate the buffer
		BufferSize = Buffer_Size;
		Buffer = new char[BufferSize];
		BytesInBuffer = 0;
		BufferPos = 0;
		WriteBuffered = false;
	}
}


PlainBufferedFile::~PlainBufferedFile()
{
	if(Buffer)
		delete[] Buffer;
}


status_t PlainBufferedFile::FlushBuffer()
//Flushes the read buffer and moves the file pointer to the position that was being read from.  It is
//automatically called when necessary when working within the file, but if you have buffered write data, call
//it yourself before deleting the PlainBufferedFile to make sure the data is written successfully (the destructor
//can't return an error code).
{
	if(WriteBuffered)
	{
		if(BFile::Write(Buffer,BufferPos) != BufferPos)
			return B_FILE_ERROR;
		BufferPos = 0;
		WriteBuffered = false;
	}
	else if(BufferPos != 0)
	{
		off_t Result = BFile::Seek(off_t(BufferPos)-off_t(BytesInBuffer),SEEK_CUR);
		if(Result == B_ERROR || Result == B_FILE_ERROR)
			return B_FILE_ERROR;
		BufferPos = 0;
	}
	return B_NO_ERROR;
}


status_t PlainBufferedFile::InitCheck()
//Returns B_NO_ERROR if the PlainBufferedFile was created successfully.  Returns B_NO_INIT if initialization of
//the underlying BFile failed.  Returns B_ERROR if reading or writing the endianness flag to or from the
//first byte of the file failed.
{
	return Valid;
}


int32 PlainBufferedFile::BufferedWrite(const void* WriteBuffer, size_t BytesToWrite)
//Writes the raw data contained in WriteBuffer.  Returns the number of bytes actually written.
{
	if(!IsWritable())
		return 0;
	size_t Counter = 0;
	if(!WriteBuffered && BufferPos != 0)
		if(FlushBuffer() != B_NO_ERROR)
			return 0;
	while(Counter < BytesToWrite)
	{
		Buffer[BufferPos++] = ((const char*)WriteBuffer)[Counter++];
		if(BufferPos == BufferSize)
		{
			//Buffer is full, need to write and reset
			int32 Result = BFile::Write(Buffer,BufferSize);
			if(Result != BufferSize)
			{
				WriteBuffered = false;
				return Counter-(BufferSize-Result);
			}
			BufferPos = 0;
		}
	}
	if(BufferPos > 0)
		WriteBuffered = true;
	return Counter;
}


int32 PlainBufferedFile::BufferedRead(void* ReadBuffer, size_t BytesToRead)
//Reads raw data into ReadBuffer.  Returns the number of bytes actually read.
{
	if(!IsReadable())
		return 0;
	if(WriteBuffered)
		if(FlushBuffer() != B_NO_ERROR)
			return 0;
	size_t Counter = 0;
	while(Counter < BytesToRead)
	{
		if(BufferPos == 0)
		{
			ssize_t Result = BFile::Read(Buffer,BufferSize);
			BytesInBuffer = Result;
			if(Result < ssize_t(BufferSize))
			{
				if(Result <= 0)
					return Counter;
			}
		}
		((char*)ReadBuffer)[Counter++] = ((const char*)Buffer)[BufferPos++];
		if(BufferPos == BytesInBuffer)
		{
			//Reached end of buffer, need to reset for another read
			if(BytesInBuffer != BufferSize)
			{
				//Already read to the end of the file
				BufferPos = 0;
				BytesInBuffer = 0;
				return Counter;
			}
			BufferPos = 0;
			BytesInBuffer = 0;
		}
	}
	return Counter;
}


ssize_t PlainBufferedFile::Read(void *buffer, size_t size)
{
	if(FlushBuffer() != B_NO_ERROR)
		return 0;
	return BFile::Read(buffer,size);
}


ssize_t PlainBufferedFile::ReadAt(off_t pos, void *buffer, size_t size)
{
	if(FlushBuffer() != B_NO_ERROR)
		return 0;
	return BFile::ReadAt(pos,buffer,size);
}


ssize_t PlainBufferedFile::Write(const void *buffer, size_t size)
{
	if(FlushBuffer() != B_NO_ERROR)
		return 0;
	return BFile::Write(buffer,size);
}


ssize_t PlainBufferedFile::WriteAt(off_t pos, const void *buffer, size_t size)
{
	if(FlushBuffer() != B_NO_ERROR)
		return 0;
	return BFile::WriteAt(pos,buffer,size);
}


off_t PlainBufferedFile::Seek(off_t position, uint32 seek_mode)
{
	if(FlushBuffer() != B_NO_ERROR)
		return 0;
	return BFile::Seek(position,seek_mode);
}


status_t PlainBufferedFile::SetSize(off_t size)
{
	if(FlushBuffer() != B_NO_ERROR)
		return B_FILE_ERROR;
	return BFile::SetSize(size);
}


status_t PlainBufferedFile::SetTo(const entry_ref *ref, uint32 open_mode)
{
	if(FlushBuffer() != B_NO_ERROR)
		return B_FILE_ERROR;
	Valid = BFile::SetTo(ref,open_mode);
	openMode = open_mode;
	if(Valid != B_NO_ERROR)
		return Valid;
	CheckNewFileStatus();
	return Valid;
}


status_t PlainBufferedFile::SetTo(const BEntry *entry, uint32 open_mode)
{
	if(FlushBuffer() != B_NO_ERROR)
		return B_FILE_ERROR;
	Valid = BFile::SetTo(entry,open_mode);
	openMode = open_mode;
	if(Valid != B_NO_ERROR)
		return Valid;
	CheckNewFileStatus();
	return Valid;
}


status_t PlainBufferedFile::SetTo(const char *path, uint32 open_mode)
{
	if(FlushBuffer() != B_NO_ERROR)
		return B_FILE_ERROR;
	Valid = BFile::SetTo(path,open_mode);
	openMode = open_mode;
	if(Valid != B_NO_ERROR)
		return Valid;
	CheckNewFileStatus();
	return Valid;
}


status_t PlainBufferedFile::SetTo(const BDirectory *dir, const char *path, uint32 open_mode)
{
	if(FlushBuffer() != B_NO_ERROR)
		return B_FILE_ERROR;
	Valid = BFile::SetTo(dir,path,open_mode);
	openMode = open_mode;
	if(Valid != B_NO_ERROR)
		return Valid;
	CheckNewFileStatus();
	return Valid;
}
